# 📁 Proje Yapısı

## Laravel API-First Backend Yapısı

```
backend/
├── app/
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── API/
│   │   │   │   ├── Auth/
│   │   │   │   │   ├── AuthController.php
│   │   │   │   │   └── RegisterController.php
│   │   │   │   ├── Tours/
│   │   │   │   │   ├── TourController.php
│   │   │   │   │   └── TourDateController.php
│   │   │   │   ├── Reservations/
│   │   │   │   │   ├── ReservationController.php
│   │   │   │   │   └── ReservationApprovalController.php
│   │   │   │   ├── Payments/
│   │   │   │   │   ├── PaymentController.php
│   │   │   │   │   ├── StripeController.php
│   │   │   │   │   └── PaymentLinkController.php
│   │   │   │   ├── Customers/
│   │   │   │   │   ├── CustomerPanelController.php
│   │   │   │   │   └── DocumentController.php
│   │   │   │   └── ...
│   │   │   └── Admin/
│   │   │       ├── ReservationManagementController.php
│   │   │       ├── TourManagementController.php
│   │   │       └── ...
│   │   ├── Middleware/
│   │   │   ├── EnsureApiToken.php
│   │   │   ├── MultiTenantMiddleware.php
│   │   │   └── ...
│   │   ├── Requests/
│   │   │   ├── API/
│   │   │   │   ├── CreateReservationRequest.php
│   │   │   │   ├── UpdateReservationRequest.php
│   │   │   │   └── ...
│   │   │   └── ...
│   │   └── Resources/
│   │       ├── TourResource.php
│   │       ├── ReservationResource.php
│   │       └── ...
│   ├── Models/
│   │   ├── User.php
│   │   ├── Brand.php
│   │   ├── Tour.php
│   │   ├── TourDate.php
│   │   ├── Reservation.php
│   │   ├── Payment.php
│   │   ├── Guest.php
│   │   ├── Room.php
│   │   └── ...
│   ├── Services/
│   │   ├── Payment/
│   │   │   ├── StripeService.php
│   │   │   ├── PaymentLinkService.php
│   │   │   └── PaymentService.php
│   │   ├── Reservation/
│   │   │   ├── ReservationService.php
│   │   │   └── ApprovalService.php
│   │   ├── Document/
│   │   │   ├── PDFService.php
│   │   │   └── QRCodeService.php
│   │   └── ...
│   ├── Events/
│   │   ├── ReservationCreated.php
│   │   ├── ReservationApproved.php
│   │   ├── PaymentCompleted.php
│   │   └── ...
│   ├── Listeners/
│   │   ├── SendReservationConfirmation.php
│   │   ├── SendPaymentLink.php
│   │   └── ...
│   └── ...
├── database/
│   ├── migrations/
│   │   ├── 2024_01_01_000001_create_brands_table.php
│   │   ├── 2024_01_01_000002_create_tours_table.php
│   │   ├── 2024_01_01_000003_create_reservations_table.php
│   │   ├── 2024_01_01_000004_create_payments_table.php
│   │   └── ...
│   ├── seeders/
│   │   ├── DatabaseSeeder.php
│   │   ├── BrandSeeder.php
│   │   ├── TourSeeder.php
│   │   └── ...
│   └── factories/
│       ├── TourFactory.php
│       ├── ReservationFactory.php
│       └── ...
├── routes/
│   ├── api.php          # API routes
│   ├── web.php          # Web routes (admin panel için)
│   └── channels.php      # Broadcasting channels
├── config/
│   ├── sanctum.php
│   ├── cashier.php
│   ├── multitenancy.php
│   └── ...
├── tests/
│   ├── Feature/
│   │   ├── Api/
│   │   │   ├── AuthTest.php
│   │   │   ├── TourTest.php
│   │   │   └── ...
│   │   └── ...
│   └── Unit/
└── ...
```

## API Route Yapısı

```php
// routes/api.php

// Public Routes
Route::post('/auth/register', [AuthController::class, 'register']);
Route::post('/auth/login', [AuthController::class, 'login']);

// Protected Routes (Sanctum)
Route::middleware('auth:sanctum')->group(function () {
    // User Routes
    Route::get('/auth/user', [AuthController::class, 'user']);
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    
    // Tours
    Route::get('/tours', [TourController::class, 'index']);
    Route::get('/tours/{id}', [TourController::class, 'show']);
    
    // Reservations
    Route::post('/reservations', [ReservationController::class, 'store']);
    Route::get('/reservations', [ReservationController::class, 'index']);
    Route::get('/reservations/{id}', [ReservationController::class, 'show']);
    
    // Payments
    Route::post('/payments/create-link', [PaymentLinkController::class, 'create']);
    Route::post('/payments/checkout', [PaymentController::class, 'checkout']);
    
    // Customer Panel
    Route::get('/customer/reservations', [CustomerPanelController::class, 'reservations']);
    Route::get('/customer/documents/{reservationId}', [DocumentController::class, 'index']);
});

// Admin Routes
Route::middleware(['auth:sanctum', 'role:admin'])->prefix('admin')->group(function () {
    Route::post('/reservations/{id}/approve', [ReservationApprovalController::class, 'approve']);
    Route::get('/reservations/pending', [ReservationManagementController::class, 'pending']);
    // ...
});

// Webhooks
Route::post('/webhooks/stripe', [StripeController::class, 'webhook']);
```

## Model İlişkileri

```php
// User Model
- hasMany(Reservation::class)
- hasMany(Payment::class)

// Tour Model
- belongsTo(Brand::class)
- hasMany(TourDate::class)
- hasMany(Reservation::class)

// Reservation Model
- belongsTo(User::class)
- belongsTo(Tour::class)
- belongsTo(TourDate::class)
- hasMany(Guest::class)
- hasMany(Room::class)
- hasMany(Payment::class)

// Payment Model
- belongsTo(Reservation::class)
- belongsTo(User::class)
```
