<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Activity;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class ActivityController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        $activities = Activity::orderBy('order')->orderBy('created_at', 'desc')->get();
        return response()->json(['data' => $activities]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'base_price' => 'required|numeric|min:0',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
            'slug' => 'nullable|string|unique:activities,slug',
            'default_start_time' => 'nullable|string', // Added
            'images' => 'nullable|array',
            'highlights' => 'nullable|array',
            'included' => 'nullable|array',
            'excluded' => 'nullable|array',
            'order' => 'integer'
        ]);

        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        $activity = Activity::create($validated);

        return response()->json([
            'message' => 'Activity created successfully',
            'data' => $activity
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id): JsonResponse
    {
        $activity = Activity::with(['dates', 'pricingTiers'])->findOrFail($id);
        return response()->json(['data' => $activity]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id): JsonResponse
    {
        $activity = Activity::findOrFail($id);

        $validated = $request->validate([
            'title' => 'string|max:255',
            'base_price' => 'numeric|min:0',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
            'slug' => 'nullable|string|unique:activities,slug,' . $id,
            'default_start_time' => 'nullable|string', // Added
            'images' => 'nullable|array',
            'highlights' => 'nullable|array',
            'included' => 'nullable|array',
            'excluded' => 'nullable|array',
            'order' => 'integer'
        ]);

        if (isset($validated['title']) && empty($request->slug) && empty($activity->slug)) {
             $validated['slug'] = Str::slug($validated['title']);
        }

        $activity->update($validated);

        return response()->json([
            'message' => 'Activity updated successfully',
            'data' => $activity
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id): JsonResponse
    {
        $activity = Activity::findOrFail($id);
        $activity->delete();

        return response()->json([
            'message' => 'Activity deleted successfully'
        ]);
    }
    
    /**
     * Store a newly created pricing tier.
     */
    public function storePricingTier(Request $request, $id): JsonResponse
    {
        $activity = Activity::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'min_age' => 'nullable|integer',
            'max_age' => 'nullable|integer',
        ]);
        
        $tier = $activity->pricingTiers()->create($validated);
        
        return response()->json([
            'message' => 'Pricing tier added successfully',
            'data' => $tier
        ], 201);
    }

    /**
     * Remove the specified pricing tier.
     */
    public function destroyPricingTier($id, $tierId): JsonResponse
    {
        $tier = \App\Models\ActivityPricingTier::where('activity_id', $id)->findOrFail($tierId);
        $tier->delete();
        
        return response()->json([
            'message' => 'Pricing tier deleted successfully'
        ]);
    }
}
