<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Payment;
use App\Models\Tour;
use App\Models\TourDate;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics
     */
    public function index(): JsonResponse
    {
        $today = Carbon::today();
        $thisMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        $lastMonthEnd = Carbon::now()->subMonth()->endOfMonth();

        // Total reservations
        $totalReservations = Reservation::count();

        // Pending approvals
        $pendingApprovals = Reservation::whereIn('status', ['pending', 'awaiting_approval'])->count();

        // Payment pending
        $paymentPending = Reservation::where('status', 'payment_pending')->count();

        // Today's tours
        $todayTours = TourDate::whereDate('start_date', $today)->count();

        // Revenue calculations
        $thisMonthRevenue = Payment::where('status', 'completed')
            ->whereBetween('paid_at', [$thisMonth, Carbon::now()])
            ->sum('amount');

        $lastMonthRevenue = Payment::where('status', 'completed')
            ->whereBetween('paid_at', [$lastMonth, $lastMonthEnd])
            ->sum('amount');

        $monthlyGrowth = $lastMonthRevenue > 0 
            ? (($thisMonthRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100 
            : 0;

        // Recent pending reservations (last 5)
        $recentPending = Reservation::with(['user', 'tour', 'tourDate'])
            ->whereIn('status', ['pending', 'awaiting_approval'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($reservation) {
                return [
                    'id' => $reservation->id,
                    'reservation_code' => $reservation->reservation_code,
                    'easy_code' => $reservation->easy_code,
                    'customer_name' => $reservation->user->name ?? $reservation->contact_info['fullName'] ?? 'N/A',
                    'tour_title' => $reservation->tour->title ?? 'N/A',
                    'total_price' => $reservation->total_price,
                    'status' => $reservation->status,
                    'created_at' => $reservation->created_at,
                ];
            });

        return response()->json([
            'data' => [
                'totalReservations' => $totalReservations,
                'pendingApprovals' => $pendingApprovals,
                'paymentPending' => $paymentPending,
                'todayTours' => $todayTours,
                'totalRevenue' => (float) $thisMonthRevenue,
                'monthlyGrowth' => round($monthlyGrowth, 2),
                'recentPending' => $recentPending,
            ]
        ]);
    }
}
