<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class TestimonialController extends Controller
{
    /**
     * Get all testimonials
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Testimonial::query();

            if ($request->has('is_active')) {
                $query->where('is_active', $request->is_active === 'true');
            }

            $testimonials = $query->with('brands')->orderBy('order')->orderBy('created_at', 'desc')->get();

            return response()->json([
                'data' => $testimonials
            ]);
        } catch (\Exception $e) {
            Log::error('Testimonials index error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get testimonial by ID
     */
    public function show(int $id): JsonResponse
    {
        try {
            $testimonial = Testimonial::with('brands')->findOrFail($id);
            return response()->json(['data' => $testimonial]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Create new testimonial
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'video_url' => 'required|string|max:500',
                'tour_name' => 'nullable|string|max:255',
                'title' => 'required|string|max:255',
                'description' => 'nullable|string',
                'order' => 'nullable|integer',
                'is_active' => 'nullable|boolean',
                'brand_ids' => 'nullable|array',
                'brand_ids.*' => 'exists:brands,id',
            ]);

            $testimonial = Testimonial::create($validated);

            // Sync brands
            if ($request->has('brand_ids')) {
                $testimonial->brands()->sync($request->brand_ids);
            }

            return response()->json([
                'data' => $testimonial->load('brands'),
                'message' => 'Testimonial created successfully'
            ], 201);
        } catch (\Exception $e) {
            Log::error('Testimonial create error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Update testimonial
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            $testimonial = Testimonial::findOrFail($id);

            $validated = $request->validate([
                'video_url' => 'sometimes|required|string|max:500',
                'tour_name' => 'nullable|string|max:255',
                'title' => 'sometimes|required|string|max:255',
                'description' => 'nullable|string',
                'order' => 'nullable|integer',
                'is_active' => 'nullable|boolean',
                'brand_ids' => 'nullable|array',
                'brand_ids.*' => 'exists:brands,id',
            ]);

            $testimonial->update($validated);

            // Sync brands
            if ($request->has('brand_ids')) {
                $testimonial->brands()->sync($request->brand_ids);
            }

            return response()->json([
                'data' => $testimonial->load('brands'),
                'message' => 'Testimonial updated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Testimonial update error: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Delete testimonial
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $testimonial = Testimonial::findOrFail($id);
            $testimonial->delete();

            return response()->json([
                'message' => 'Testimonial deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
