<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Tour;
use App\Models\TourDate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;

class TourController extends Controller
{
    /**
     * Get all tours (admin)
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = Tour::select([
                'id', 'brand_id', 'code', 'title', 'subtitle', 'type', 'nights', 
                'image', 'price', 'discounted_price', 'is_active', 'order', 
                'label', 'seo_title', 'slug', 'seo_description', 'seo_image',
                'created_at', 'updated_at'
            ])->with(['brand', 'dates', 'categories']);

            if ($request->has('brand_id')) {
                $query->where('brand_id', $request->brand_id);
            }

            if ($request->has('type')) {
                $query->where('type', $request->type);
            }

            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('code', 'like', "%{$search}%")
                      ->orWhere('subtitle', 'like', "%{$search}%");
                });
            }

            try {
                $tours = $query->orderBy('order')
                    ->orderBy('created_at', 'desc')
                    ->paginate($request->get('per_page', 15));
            } catch (\Exception $e) {
                Log::warning('Order column issue, using created_at only: ' . $e->getMessage());
                $tours = $query->orderBy('created_at', 'desc')
                    ->paginate($request->get('per_page', 15));
            }

            return response()->json($tours);
        } catch (\Exception $e) {
            Log::error('Tour index error: ' . $e->getMessage());
            return response()->json([
                'error' => 'Turlar yüklenirken hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tour by ID (admin)
     */
    public function show(int $id): JsonResponse
    {
        try {
            $tour = Tour::with(['brand', 'dates', 'categories'])
                ->findOrFail($id);

            $tourArray = $tour->toArray();
            
            // Dates ilişkisini manuel olarak işle - single_supplement_price dahil et
            if (isset($tourArray['dates']) && is_array($tourArray['dates']) && $tour->dates) {
                $tourArray['dates'] = $tour->dates->map(function ($dateModel) {
                    $dateArray = $dateModel->toArray();
                    
                    // single_supplement_price: DB'den direkt oku (cast bypass)
                    $dbValue = DB::table('tour_dates')
                        ->where('id', $dateModel->id)
                        ->value('single_supplement_price');
                    
                    // Değeri string olarak ekle (null ise null, 0 ise '0', diğerleri string)
                    // PHP'de 0, 0.0, '0', '0.00' gibi değerler farklı şekillerde gelebilir
                    if ($dbValue !== null) {
                        // Numerik değer mi kontrol et
                        if (is_numeric($dbValue)) {
                            $dateArray['single_supplement_price'] = (string)floatval($dbValue);
                        } else {
                            $dateArray['single_supplement_price'] = null;
                        }
                    } else {
                        $dateArray['single_supplement_price'] = null;
                    }
                    
                    // CRITICAL: transportation ve hotel_options alanlarını ekle
                    // Bu alanlar JSON olarak saklanıyor, Eloquent cast ediyor
                    $dateArray['transportation'] = $dateModel->transportation;
                    $dateArray['hotel_options'] = $dateModel->hotel_options;
                    $dateArray['bus_layout'] = $dateModel->bus_layout;
                    $dateArray['meal_list'] = $dateModel->meal_list;
                    $dateArray['meal_options'] = $dateModel->meal_options;
                    if (empty($dateArray['meal_options']) && !empty($dateModel->meal_list)) {
                        $dateArray['meal_options'] = ['categories' => [['key' => 'main', 'label' => 'Ana Menü', 'options' => $dateModel->meal_list]]];
                    }
                    // Doluluk: sadece iptal edilmemiş rezervasyonlardan hesapla
                    $reservations = Reservation::where('tour_date_id', $dateModel->id)
                        ->whereNotIn('status', ['cancelled'])
                        ->with('guests')
                        ->get();
                    $bookedSeats = [];
                    $guestsWithoutSeat = 0;
                    foreach ($reservations as $r) {
                        foreach ($r->guests as $g) {
                            $hasSeat = false;
                            if ($g->seat_number !== null && $g->seat_number !== '') {
                                $bookedSeats[] = (int) $g->seat_number;
                                $hasSeat = true;
                            }
                            if ($g->extra_seat_number !== null && $g->extra_seat_number !== '') {
                                $bookedSeats[] = (int) $g->extra_seat_number;
                                $hasSeat = true;
                            }
                            if (!empty($g->departure_seat_number)) {
                                $bookedSeats[] = (int) $g->departure_seat_number;
                                $hasSeat = true;
                            }
                            if (!empty($g->return_seat_number)) {
                                $bookedSeats[] = (int) $g->return_seat_number;
                                $hasSeat = true;
                            }
                            if (!$hasSeat) {
                                $guestsWithoutSeat++;
                            }
                        }
                    }
                    $dateArray['booked_seat_numbers'] = array_values(array_unique($bookedSeats));
                    // Doluluk = satılan koltuk sayısı (kişi sayısı değil)
                    $dateArray['booked'] = count($dateArray['booked_seat_numbers']);
                    $dateArray['guests_without_seat'] = $guestsWithoutSeat;
                    return $dateArray;
                })->toArray();
            }
            
            // JSON alanlarını string olarak döndür
            $jsonFields = ['description', 'itinerary', 'images', 'included', 'excluded', 'highlights', 'pickup_points', 'departure_point', 'flight_options', 'room_options', 'hotel_options', 'feature_cards', 'countries', 'seo_title', 'label', 'extra_services', 'merchandise_items', 'cancellation_insurance', 'bus_seat_layout'];
            foreach ($jsonFields as $field) {
                if (isset($tourArray[$field])) {
                    if (is_array($tourArray[$field])) {
                        $tourArray[$field] = json_encode($tourArray[$field]);
                    } elseif ($tourArray[$field] === null) {
                        if ($field === 'pickup_points') {
                            $tourArray[$field] = json_encode([]);
                        } else {
                            $tourArray[$field] = null;
                        }
                    }
                } else {
                    if ($field === 'pickup_points') {
                        $tourArray[$field] = json_encode([]);
                    }
                }
            }
            
            // SEO alanlarının response'da olduğundan emin ol
            if (!isset($tourArray['slug'])) $tourArray['slug'] = null;
            if (!isset($tourArray['seo_description'])) $tourArray['seo_description'] = null;
            if (!isset($tourArray['seo_image'])) $tourArray['seo_image'] = null;
            if (!isset($tourArray['seo_title'])) $tourArray['seo_title'] = null;
            if (!isset($tourArray['subtitle'])) $tourArray['subtitle'] = null;

            return response()->json([
                'data' => $tourArray
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create new tour (admin)
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'code' => 'required|string|unique:tours,code',
            'title' => 'required|string',
            'type' => 'required|in:bus,flight,event,vacation,car_rental,umre',
            'brand_id' => 'nullable|exists:brands,id',
        ]);

        $tourData = $request->all();
        if (!isset($tourData['is_active'])) {
            $tourData['is_active'] = true;
        }

        // Auto-generate slug if empty
        if (empty($tourData['slug']) && !empty($tourData['title'])) {
            $titleForSlug = '';
            if (is_array($tourData['title'])) {
                $titleForSlug = $tourData['title']['tr'] ?? $tourData['title']['en'] ?? reset($tourData['title']);
            } else {
                $titleForSlug = $tourData['title'];
            }
            $tourData['slug'] = \Illuminate\Support\Str::slug($titleForSlug);
        }

        // flight_options kolonu yoksa tourData'dan çıkar (migration çalıştırılmamış olabilir)
        if (isset($tourData['flight_options']) && !Schema::hasColumn('tours', 'flight_options')) {
            unset($tourData['flight_options']);
        }

        $tour = Tour::create($tourData);

        if ($request->has('dates')) {
            foreach ($request->dates as $dateData) {
                // single_supplement_price: string'den float'a çevir
                if (isset($dateData['single_supplement_price']) && $dateData['single_supplement_price'] !== null && $dateData['single_supplement_price'] !== '') {
                    $dateData['single_supplement_price'] = (float)$dateData['single_supplement_price'];
                } else {
                    $dateData['single_supplement_price'] = null;
                }
                
                // CRITICAL: Transportation field handling (flight options, airlines, etc.)
                if (isset($dateData['transportation'])) {
                    if (is_string($dateData['transportation'])) {
                        // Decode JSON string to array
                        $decoded = json_decode($dateData['transportation'], true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                            $dateData['transportation'] = $decoded;
                        } else {
                            // Invalid JSON, set to null
                            $dateData['transportation'] = null;
                        }
                    }
                    // If already array, leave as is (Eloquent will cast to JSON)
                }
                
                // CRITICAL: Hotel options field handling
                if (isset($dateData['hotel_options'])) {
                    if (is_string($dateData['hotel_options'])) {
                        // Decode JSON string to array
                        $decoded = json_decode($dateData['hotel_options'], true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                            $dateData['hotel_options'] = $decoded;
                        } else {
                            // Invalid JSON, set to null
                            $dateData['hotel_options'] = null;
                        }
                    }
                    // If already array, leave as is (Eloquent will cast to JSON)
                }
                
                // CRITICAL: Bus layout field handling
            if (isset($dateData['bus_layout']) || isset($dateData['bus_seat_layout'])) {
                $rawBusLayout = $dateData['bus_layout'] ?? $dateData['bus_seat_layout'];
                if (is_string($rawBusLayout)) {
                    $decoded = json_decode($rawBusLayout, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        $dateData['bus_layout'] = $decoded;
                    }
                } else if (is_array($rawBusLayout)) {
                    $dateData['bus_layout'] = $rawBusLayout;
                }
            }
            if (isset($dateData['meal_list'])) {
                if (is_string($dateData['meal_list'])) {
                    $decoded = json_decode($dateData['meal_list'], true);
                    $dateData['meal_list'] = (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) ? $decoded : [];
                } elseif (!is_array($dateData['meal_list'])) {
                    $dateData['meal_list'] = [];
                }
            }
            if (isset($dateData['meal_options'])) {
                if (is_string($dateData['meal_options'])) {
                    $decoded = json_decode($dateData['meal_options'], true);
                    $dateData['meal_options'] = (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) ? $decoded : null;
                } elseif (!is_array($dateData['meal_options'])) {
                    $dateData['meal_options'] = null;
                }
            }
            
            $tour->dates()->create($dateData);
            }
        }

        if ($request->has('category_ids')) {
            $tour->categories()->sync($request->category_ids);
        }

        return response()->json([
            'data' => $tour->load(['brand', 'dates', 'categories'])
        ], 201);
    }

    /**
     * Update tour (admin)
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $tour = Tour::findOrFail($id);

        DB::beginTransaction();
        try {
            $dates = $request->input('dates');
            $categoryIds = $request->input('category_ids');
            $termsIds = $request->input('terms_ids');
            $bankAccountIds = $request->input('bank_account_ids');
            
            $updateData = $request->except(['dates', 'category_ids', 'terms_ids', 'bank_account_ids']);
            
            if ($request->has('pickup_points')) {
                $updateData['pickup_points'] = $request->input('pickup_points');
            }
            
            // Fiyat alanlarını numeric'e çevir
            if (isset($updateData['price'])) {
                $updateData['price'] = is_numeric($updateData['price']) ? (float)$updateData['price'] : 0;
            }
            if (isset($updateData['discounted_price'])) {
                $updateData['discounted_price'] = is_numeric($updateData['discounted_price']) ? (float)$updateData['discounted_price'] : null;
            }
            
            if (isset($updateData['nights'])) {
                $updateData['nights'] = is_numeric($updateData['nights']) ? (int)$updateData['nights'] : 0;
            }
            
            // flight_options kolonu yoksa updateData'dan çıkar (migration çalıştırılmamış olabilir)
            if (isset($updateData['flight_options']) && !Schema::hasColumn('tours', 'flight_options')) {
                unset($updateData['flight_options']);
            }
            
            // JSON alanlarını işle
            $jsonFields = ['title', 'subtitle', 'description', 'itinerary', 'images', 'included', 'excluded', 'highlights', 'departure_point', 'flight_options', 'room_options', 'hotel_options', 'feature_cards', 'countries', 'seo_title', 'label', 'extra_services', 'merchandise_items', 'cancellation_insurance', 'bus_seat_layout'];
            foreach ($jsonFields as $field) {
                if (isset($updateData[$field])) {
                    if (is_array($updateData[$field])) {
                        // Zaten array, olduğu gibi bırak
                    } elseif (is_string($updateData[$field])) {
                        \Illuminate\Support\Facades\Log::info('Processing JSON field', [
                            'field' => $field,
                            'value' => $updateData[$field],
                            'type' => gettype($updateData[$field])
                        ]);
                        
                        $decoded = json_decode($updateData[$field], true);
                        if (json_last_error() === JSON_ERROR_NONE && (is_array($decoded) || is_object($decoded))) {
                            $updateData[$field] = $decoded;
                        } else {
                            // Decode edilemezse string olarak bırak (belki normal text'tir)
                             \Illuminate\Support\Facades\Log::warning('Failed to decode JSON field', [
                                'field' => $field, 
                                'value' => $updateData[$field]
                            ]);
                        }
                    }
                }
            }
            
            // pickup_points için özel işlem: Array ise direkt ata, JSON string ise decode et (Eloquent array cast ediyor)
            if (isset($updateData['pickup_points'])) {
                if (is_string($updateData['pickup_points'])) {
                    $decoded = json_decode($updateData['pickup_points'], true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        $updateData['pickup_points'] = $decoded;
                    }
                }
                // Array ise zaten dokunma, Eloquent halleder.
            } else if ($request->has('pickup_points') && $request->input('pickup_points') === null) {
                $updateData['pickup_points'] = []; // Null gelirse boş array yap
            }
            
            // SEO alanlarını ekle
            if ($request->has('slug')) {
                $updateData['slug'] = $request->input('slug');
            }
            
            // Auto-generate slug if it's empty in request OR if it's currently empty in DB and not provided
            if (empty($updateData['slug'])) {
                // If provided but empty string, or not provided but we want to fill it
                $titleForSlug = '';
                // Consider the title from request if available, otherwise from existing tour
                $titleSource = isset($updateData['title']) ? $updateData['title'] : $tour->title;
                
                if (is_array($titleSource)) {
                    $titleForSlug = $titleSource['tr'] ?? $titleSource['en'] ?? reset($titleSource);
                } else {
                    $titleForSlug = $titleSource;
                }
                
                if (!empty($titleForSlug)) {
                    $updateData['slug'] = \Illuminate\Support\Str::slug($titleForSlug);
                }
            }

            if ($request->has('seo_description')) {
                $updateData['seo_description'] = $request->input('seo_description');
            }
            if ($request->has('seo_image')) {
                $updateData['seo_image'] = $request->input('seo_image');
            }
            if ($request->has('subtitle')) {
                $updateData['subtitle'] = $request->input('subtitle');
            }
            if ($request->has('seo_title')) {
                $updateData['seo_title'] = $request->input('seo_title');
            }
            
            // Debug: departure_point değerini logla
            \Illuminate\Support\Facades\Log::info('Updating tour departure_point', [
                'tour_id' => $tour->id,
                'departure_point_before' => $tour->departure_point,
                'departure_point_in_updateData' => $updateData['departure_point'] ?? 'NOT_SET',
                'all_update_data_keys' => array_keys($updateData)
            ]);
            
            $tour->update($updateData);
            $tour->refresh();
            
            // Debug: güncelleme sonrası departure_point değerini logla
            \Illuminate\Support\Facades\Log::info('After update departure_point', [
                'tour_id' => $tour->id,
                'departure_point_after' => $tour->departure_point
            ]);

            // Dates güncelle
            if ($dates !== null) {
                $tour->dates()->delete();
                foreach ($dates as $dateData) {
                    // Fiyat alanlarını float'a çevir
                    $dateData['adult_price'] = isset($dateData['adult_price']) ? (float)$dateData['adult_price'] : 0;
                    $dateData['adult_discounted_price'] = isset($dateData['adult_discounted_price']) ? (float)$dateData['adult_discounted_price'] : 0;
                    $dateData['child_price'] = isset($dateData['child_price']) ? (float)$dateData['child_price'] : 0;
                    $dateData['child_discounted_price'] = isset($dateData['child_discounted_price']) ? (float)$dateData['child_discounted_price'] : 0;
                    $dateData['baby_price'] = isset($dateData['baby_price']) ? (float)$dateData['baby_price'] : 0;
                    $dateData['baby_discounted_price'] = isset($dateData['baby_discounted_price']) ? (float)$dateData['baby_discounted_price'] : 0;
                    $dateData['capacity'] = isset($dateData['capacity']) ? (int)$dateData['capacity'] : 40;
                    $dateData['booked'] = isset($dateData['booked']) ? (int)$dateData['booked'] : 0;
                    
                    // single_supplement_price: string'den float'a çevir
                    if (isset($dateData['single_supplement_price']) && $dateData['single_supplement_price'] !== null && $dateData['single_supplement_price'] !== '') {
                        $dateData['single_supplement_price'] = (float)$dateData['single_supplement_price'];
                    } else {
                        $dateData['single_supplement_price'] = null;
                    }
                    
                    // CRITICAL: Transportation field handling (flight options, airlines, etc.)
                    if (isset($dateData['transportation'])) {
                        if (is_string($dateData['transportation'])) {
                            // Decode JSON string to array
                            $decoded = json_decode($dateData['transportation'], true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                                $dateData['transportation'] = $decoded;
                            } else {
                                // Invalid JSON, set to null
                                $dateData['transportation'] = null;
                            }
                        }
                        // If already array, leave as is (Eloquent will cast to JSON)
                    }
                    
                    // CRITICAL: Hotel options field handling
                    if (isset($dateData['hotel_options'])) {
                        if (is_string($dateData['hotel_options'])) {
                            // Decode JSON string to array
                            $decoded = json_decode($dateData['hotel_options'], true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                                $dateData['hotel_options'] = $decoded;
                            } else {
                                // Invalid JSON, set to null
                                $dateData['hotel_options'] = null;
                            }
                        }
                        // If already array, leave as is (Eloquent will cast to JSON)
                    }

                    // CRITICAL: Bus layout field handling
                    if (isset($dateData['bus_layout']) || isset($dateData['bus_seat_layout'])) {
                        $rawBusLayout = $dateData['bus_layout'] ?? $dateData['bus_seat_layout'];
                        if (is_string($rawBusLayout)) {
                            $decoded = json_decode($rawBusLayout, true);
                            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                                $dateData['bus_layout'] = $decoded;
                            }
                        } else if (is_array($rawBusLayout)) {
                            $dateData['bus_layout'] = $rawBusLayout;
                        }
                    }
                    if (isset($dateData['meal_list'])) {
                        if (is_string($dateData['meal_list'])) {
                            $decoded = json_decode($dateData['meal_list'], true);
                            $dateData['meal_list'] = (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) ? $decoded : [];
                        } elseif (!is_array($dateData['meal_list'])) {
                            $dateData['meal_list'] = [];
                        }
                    }
                    if (isset($dateData['meal_options'])) {
                        if (is_string($dateData['meal_options'])) {
                            $decoded = json_decode($dateData['meal_options'], true);
                            $dateData['meal_options'] = (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) ? $decoded : null;
                        } elseif (!is_array($dateData['meal_options'])) {
                            $dateData['meal_options'] = null;
                        }
                    }
                    
                    $createdDate = $tour->dates()->create($dateData);
                    
                    // GARANTİ: DB'ye direkt yaz ve doğrula
                    if (isset($dateData['single_supplement_price']) && $dateData['single_supplement_price'] !== null) {
                        DB::table('tour_dates')
                            ->where('id', $createdDate->id)
                            ->update(['single_supplement_price' => (float)$dateData['single_supplement_price']]);
                    }
                }
            }

            // Categories güncelle
            if ($categoryIds !== null) {
                $tour->categories()->sync($categoryIds);
            }

            // Terms güncelle
            if ($termsIds !== null) {
                $tour->update(['terms_ids' => $termsIds]);
            }

            // Bank Accounts güncelle
            if ($bankAccountIds !== null) {
                $tour->update(['bank_account_ids' => $bankAccountIds]);
            }

            DB::commit();

            // Tour'u yeniden yükle ve döndür
            $tour = $tour->fresh()->load(['brand', 'dates', 'categories']);
            $tourArray = $tour->toArray();
            
            // Dates ilişkisini manuel olarak işle - single_supplement_price dahil et
            if (isset($tourArray['dates']) && is_array($tourArray['dates']) && $tour->dates) {
                $tourArray['dates'] = $tour->dates->map(function ($dateModel) {
                    $dateArray = $dateModel->toArray();
                    
                    // single_supplement_price: DB'den direkt oku (cast bypass)
                    $dbValue = DB::table('tour_dates')
                        ->where('id', $dateModel->id)
                        ->value('single_supplement_price');
                    
                    // Değeri string olarak ekle
                    // PHP'de 0, 0.0, '0', '0.00' gibi değerler farklı şekillerde gelebilir
                    if ($dbValue !== null) {
                        // Numerik değer mi kontrol et
                        if (is_numeric($dbValue)) {
                            $dateArray['single_supplement_price'] = (string)floatval($dbValue);
                        } else {
                            $dateArray['single_supplement_price'] = null;
                        }
                    } else {
                        $dateArray['single_supplement_price'] = null;
                    }
                    
                    $dateArray['transportation'] = $dateModel->transportation;
                    $dateArray['hotel_options'] = $dateModel->hotel_options;
                    $dateArray['bus_layout'] = $dateModel->bus_layout;
                    $dateArray['meal_list'] = $dateModel->meal_list;
                    $dateArray['meal_options'] = $dateModel->meal_options;
                    if (empty($dateArray['meal_options']) && !empty($dateModel->meal_list)) {
                        $dateArray['meal_options'] = ['categories' => [['key' => 'main', 'label' => 'Ana Menü', 'options' => $dateModel->meal_list]]];
                    }
                    // Doluluk: sadece iptal edilmemiş rezervasyonlardan hesapla
                    $reservations = Reservation::where('tour_date_id', $dateModel->id)
                        ->whereNotIn('status', ['cancelled'])
                        ->with('guests')
                        ->get();
                    $bookedSeats = [];
                    $guestsWithoutSeat = 0;
                    foreach ($reservations as $r) {
                        foreach ($r->guests as $g) {
                            $hasSeat = false;
                            if ($g->seat_number !== null && $g->seat_number !== '') {
                                $bookedSeats[] = (int) $g->seat_number;
                                $hasSeat = true;
                            }
                            if ($g->extra_seat_number !== null && $g->extra_seat_number !== '') {
                                $bookedSeats[] = (int) $g->extra_seat_number;
                                $hasSeat = true;
                            }
                            if (!empty($g->departure_seat_number)) {
                                $bookedSeats[] = (int) $g->departure_seat_number;
                                $hasSeat = true;
                            }
                            if (!empty($g->return_seat_number)) {
                                $bookedSeats[] = (int) $g->return_seat_number;
                                $hasSeat = true;
                            }
                            if (!$hasSeat) {
                                $guestsWithoutSeat++;
                            }
                        }
                    }
                    $dateArray['booked_seat_numbers'] = array_values(array_unique($bookedSeats));
                    // Doluluk = satılan koltuk sayısı (kişi sayısı değil)
                    $dateArray['booked'] = count($dateArray['booked_seat_numbers']);
                    $dateArray['guests_without_seat'] = $guestsWithoutSeat;
                    return $dateArray;
                })->toArray();
            }
            
            // JSON alanlarını string olarak döndür
            $jsonFields = ['description', 'itinerary', 'images', 'included', 'excluded', 'highlights', 'pickup_points', 'departure_point', 'flight_options', 'room_options', 'hotel_options', 'feature_cards', 'countries', 'label', 'extra_services', 'merchandise_items', 'cancellation_insurance', 'bus_seat_layout'];
            foreach ($jsonFields as $field) {
                if (isset($tourArray[$field])) {
                    if (is_array($tourArray[$field])) {
                        $tourArray[$field] = json_encode($tourArray[$field]);
                    } elseif ($tourArray[$field] === null) {
                        if ($field === 'pickup_points') {
                            $tourArray[$field] = json_encode([]);
                        } else {
                            $tourArray[$field] = null;
                        }
                    }
                } else {
                    if ($field === 'pickup_points') {
                        $tourArray[$field] = json_encode([]);
                    }
                }
            }
            
            // SEO alanlarının response'da olduğundan emin ol
            if (!isset($tourArray['slug'])) $tourArray['slug'] = null;
            if (!isset($tourArray['seo_description'])) $tourArray['seo_description'] = null;
            if (!isset($tourArray['seo_image'])) $tourArray['seo_image'] = null;
            if (!isset($tourArray['seo_title'])) $tourArray['seo_title'] = null;
            if (!isset($tourArray['subtitle'])) $tourArray['subtitle'] = null;

            return response()->json([
                'data' => $tourArray
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Tour update error: ' . $e->getMessage());
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
