<?php

namespace App\Http\Controllers\Api\Errors;

use App\Http\Controllers\Controller;
use App\Models\ErrorLog;
use App\Services\ErrorLogService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ErrorLogController extends Controller
{
    /**
     * Log an error from frontend
     */
    public function log(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'message' => 'required|string',
            'details' => 'nullable|array',
            'stack' => 'nullable|string',
            'endpoint' => 'nullable|string',
            'method' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $errorLog = ErrorLogService::logFrontend(
            $request->message,
            array_merge($request->details ?? [], [
                'stack' => $request->stack,
                'status' => $request->details['status'] ?? 500,
                'request_data' => $request->details['request_data'] ?? [],
            ]),
            $request->endpoint ?? $request->details['endpoint'] ?? null,
            $request
        );

        return response()->json([
            'message' => 'Error logged successfully',
            'error_log_id' => $errorLog?->id ?? null,
        ], 201);
    }

    /**
     * Get error logs (admin only)
     */
    public function index(Request $request): JsonResponse
    {
        // Check if table exists
        if (!\Illuminate\Support\Facades\Schema::hasTable('error_logs')) {
            return response()->json([
                'data' => [],
                'current_page' => 1,
                'last_page' => 1,
                'per_page' => 20,
                'total' => 0,
                'message' => 'error_logs table does not exist. Please run migrations.'
            ], 200);
        }

        $query = ErrorLog::query()->orderBy('created_at', 'desc');

        // Filters
        if ($request->has('source')) {
            $query->where('source', $request->source);
        }

        if ($request->has('resolved')) {
            $query->where('resolved', filter_var($request->resolved, FILTER_VALIDATE_BOOLEAN));
        }

        if ($request->has('error_type')) {
            $query->where('error_type', 'like', '%' . $request->error_type . '%');
        }

        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        $perPage = $request->get('per_page', 20);
        $errors = $query->with(['user', 'resolver'])->paginate($perPage);

        return response()->json($errors);
    }

    /**
     * Get error log details
     */
    public function show(int $id): JsonResponse
    {
        if (!\Illuminate\Support\Facades\Schema::hasTable('error_logs')) {
            return response()->json([
                'message' => 'error_logs table does not exist. Please run migrations.'
            ], 404);
        }

        $errorLog = ErrorLog::with(['user', 'resolver'])->findOrFail($id);

        return response()->json($errorLog);
    }

    /**
     * Mark error as resolved
     */
    public function resolve(Request $request, int $id): JsonResponse
    {
        $errorLog = ErrorLog::findOrFail($id);

        $errorLog->update([
            'resolved' => true,
            'resolved_at' => now(),
            'resolved_by' => $request->user()->id,
            'notes' => $request->notes,
        ]);

        return response()->json([
            'message' => 'Error marked as resolved',
            'error_log' => $errorLog->load(['user', 'resolver']),
        ]);
    }

    /**
     * Delete error log
     */
    public function destroy(int $id): JsonResponse
    {
        $errorLog = ErrorLog::findOrFail($id);
        $errorLog->delete();

        return response()->json([
            'message' => 'Error log deleted successfully',
        ]);
    }
}
