<?php

namespace App\Http\Controllers\API\Invoices;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Reservation;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\StreamedResponse;

class InvoiceController extends Controller
{
    /**
     * List invoices
     */
    public function index(Request $request): JsonResponse
    {
        $query = Invoice::with(['reservation.tour', 'reservation.user'])
            ->orderBy('created_at', 'desc');

        if ($request->has('date_from')) {
            $query->where('invoice_date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->where('invoice_date', '<=', $request->date_to);
        }
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $invoices = $query->paginate($request->get('per_page', 15));

        return response()->json($invoices);
    }

    /**
     * Generate an invoice for a reservation
     */
    public function generate(Request $request): JsonResponse
    {
        $request->validate([
            'reservation_id' => 'required|exists:reservations,id',
        ]);

        $reservation = Reservation::with(['tour', 'tourDate', 'guests', 'user'])->findOrFail($request->reservation_id);

        // Check if invoice already exists for this reservation
        $existingInvoice = Invoice::where('reservation_id', $reservation->id)->first();
        if ($existingInvoice) {
            return response()->json([
                'message' => 'Bu rezervasyon için zaten bir fatura kesilmiş.',
                'invoice' => $existingInvoice
            ], 422);
        }

        try {
            DB::beginTransaction();

            $invoiceNo = $this->generateInvoiceNumber();

            // Prepare billing details from contact info or user
            $contactInfo = $reservation->contact_info;
            $billingDetails = [
                'name' => $contactInfo['fullName'] ?? $reservation->user?->name ?? '–',
                'email' => $contactInfo['email'] ?? $reservation->user?->email ?? '–',
                'phone' => $contactInfo['phone'] ?? $reservation->user?->phone ?? '–',
                'address' => $contactInfo['address'] ?? '–',
                'city' => $contactInfo['city'] ?? '–',
                'zip_code' => $contactInfo['zipCode'] ?? $contactInfo['zip_code'] ?? '–',
                'country' => $contactInfo['country'] ?? '–',
                'invoice_type' => $contactInfo['invoiceType'] ?? 'individual',
                'company_name' => $contactInfo['companyName'] ?? null,
                'tax_id' => $contactInfo['taxID'] ?? $contactInfo['tax_id'] ?? null,
            ];

            // Prepare line items
            $items = [
                [
                    'description' => ($reservation->tour->title['tr'] ?? $reservation->tour->title) . ' - ' . ($reservation->tour_date->start_date ?? '–'),
                    'quantity' => count($reservation->guests),
                    'unit_price' => $reservation->total_price / (count($reservation->guests) ?: 1),
                    'total' => $reservation->total_price,
                ]
            ];

            $invoice = Invoice::create([
                'reservation_id' => $reservation->id,
                'invoice_no' => $invoiceNo,
                'invoice_date' => now()->toDateString(),
                'total_price' => $reservation->total_price,
                'discount_amount' => $reservation->discount_amount ?? 0,
                'tax_amount' => 0, // Could be calculated if needed
                'status' => $reservation->remaining_amount <= 0 ? 'paid' : 'pending',
                'billing_details' => $billingDetails,
                'items' => $items,
                'notes' => $request->notes,
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Fatura başarıyla oluşturuldu.',
                'invoice' => $invoice
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Fatura oluşturulamadı.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download invoice PDF
     */
    public function download($id)
    {
        $invoice = Invoice::with(['reservation.tour', 'reservation.guests'])->findOrFail($id);
        
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('invoices.pdf', compact('invoice'));
        $pdf->setOption(['isRemoteEnabled' => true]);
        
        return $pdf->download("fatura-{$invoice->invoice_no}.pdf");
    }

    /**
     * Export invoices to CSV
     */
    public function exportCsv(Request $request): StreamedResponse
    {
        $query = Invoice::with(['reservation.tour', 'reservation.user'])
            ->orderBy('created_at', 'desc');

        if ($request->has('date_from')) {
            $query->where('invoice_date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->where('invoice_date', '<=', $request->date_to);
        }

        $invoices = $query->get();

        $headers = [
            "Content-type"        => "text/csv; charset=UTF-8",
            "Content-Disposition" => "attachment; filename=faturalar_" . date('Y-m-d') . ".csv",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        ];

        $callback = function() use($invoices) {
            $file = fopen('php://output', 'w');
            // UTF-8 BOM for Excel
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            fputcsv($file, [
                'Fatura No', 'Tarih', 'Rezervasyon Kodu', 'Müşteri', 'Ürün', 'Tutar', 'Durum'
            ], ';');

            foreach ($invoices as $invoice) {
                fputcsv($file, [
                    $invoice->invoice_no,
                    $invoice->invoice_date->toDateString(),
                    $invoice->reservation->reservation_code,
                    $invoice->billing_details['name'] ?? '–',
                    $invoice->items[0]['description'] ?? '–',
                    $invoice->total_price,
                    $invoice->status === 'paid' ? 'Ödendi' : 'Bekliyor',
                ], ';');
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Cancel (delete) an invoice
     */
    public function cancel($id): JsonResponse
    {
        $invoice = Invoice::findOrFail($id);
        
        // Prevent cancelling paid invoices if needed, but for now allow admin to do whatever
        // if ($invoice->status === 'paid') { ... }

        $invoice->delete();

        return response()->json([
            'message' => 'Fatura başarıyla iptal edildi ve silindi.',
        ]);
    }

    /**
     * Generate sequential invoice number
     */
    private function generateInvoiceNumber(): string
    {
        $year = date('Y');
        
        // Find the last invoice number for the current year
        $lastInvoice = Invoice::where('invoice_no', 'like', "AVG{$year}%")
            ->orderBy('invoice_no', 'desc')
            ->first();

        if (!$lastInvoice) {
            return "AVG{$year}00001";
        }

        // Extract the sequence part (last 5 digits)
        $lastNumber = (int) substr($lastInvoice->invoice_no, 7);
        $newNumber = str_pad($lastNumber + 1, 5, '0', STR_PAD_LEFT);

        return "AVG{$year}{$newNumber}";
    }
}
