<?php

namespace App\Http\Controllers\API\Payments;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\PaymentLink;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Stripe\StripeClient;
use Illuminate\Support\Str;

class PaymentLinkController extends Controller
{
    private StripeClient $stripe;

    public function __construct()
    {
        $this->stripe = new StripeClient(config('services.stripe.secret'));
    }

    /**
     * Create payment link
     */
    public function create(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'reservation_id' => 'required|exists:reservations,id',
            'amount' => 'required|numeric|min:0.01',
            'expires_in_days' => 'nullable|integer|min:1|max:30',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $reservation = Reservation::findOrFail($request->reservation_id);

        // Check if user has permission (admin or reservation owner)
        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            // Create Stripe payment link
            $stripeLink = $this->stripe->paymentLinks->create([
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'eur',
                        'product_data' => [
                            'name' => "Rezervasyon Ödemesi: {$reservation->reservation_code}",
                        ],
                        'unit_amount' => $request->amount * 100,
                    ],
                    'quantity' => 1,
                ]],
                'metadata' => [
                    'reservation_id' => $reservation->id,
                ],
            ]);

            // Create payment link record
            $linkId = 'PL-' . strtoupper(Str::random(12));
            $expiresAt = $request->expires_in_days 
                ? now()->addDays($request->expires_in_days)
                : now()->addDays(7);

            $paymentLink = PaymentLink::create([
                'reservation_id' => $reservation->id,
                'link_id' => $linkId,
                'stripe_payment_link_id' => $stripeLink->id,
                'amount' => $request->amount,
                'status' => 'active',
                'expires_at' => $expiresAt,
            ]);

            return response()->json([
                'payment_link' => [
                    'id' => $paymentLink->id,
                    'link_id' => $paymentLink->link_id,
                    'url' => $stripeLink->url,
                    'amount' => $paymentLink->amount,
                    'expires_at' => $paymentLink->expires_at->toIso8601String(),
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Payment link creation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment link
     */
    public function show(Request $request, string $linkId): JsonResponse
    {
        $paymentLink = PaymentLink::where('link_id', $linkId)
            ->orWhere('id', $linkId)
            ->firstOrFail();

        $reservation = $paymentLink->reservation;

        // Check if user has permission
        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        // Get Stripe link URL
        $stripeUrl = null;
        if ($paymentLink->stripe_payment_link_id) {
            try {
                $stripeLink = $this->stripe->paymentLinks->retrieve($paymentLink->stripe_payment_link_id);
                $stripeUrl = $stripeLink->url;
            } catch (\Exception $e) {
                // Ignore errors
            }
        }

        return response()->json([
            'payment_link' => [
                'id' => $paymentLink->id,
                'link_id' => $paymentLink->link_id,
                'url' => $stripeUrl,
                'amount' => $paymentLink->amount,
                'status' => $paymentLink->status,
                'expires_at' => $paymentLink->expires_at?->toIso8601String(),
                'reservation' => [
                    'id' => $reservation->id,
                    'reservation_code' => $reservation->reservation_code,
                    'total_price' => $reservation->total_price,
                    'paid_amount' => $reservation->paid_amount,
                    'remaining_amount' => $reservation->remaining_amount,
                ]
            ]
        ]);
    }
}
