<?php

namespace App\Http\Controllers\API\Reservations;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Tour;
use App\Models\TourDate;
use App\Models\Guest;
use App\Models\Room;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ReservationController extends Controller
{
    /**
     * Create a new reservation
     */
    public function store(Request $request): JsonResponse
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'tour_id' => 'required|exists:tours,id',
            'tour_date_id' => 'required|exists:tour_dates,id',
            'guests' => 'required|array|min:1',
            'guests.*.first_name' => 'required|string',
            'guests.*.last_name' => 'required|string',
            'guests.*.type' => 'required|in:adult,student,child,baby',
            'rooms' => 'required|array|min:1',
            'contact_info' => 'required|array',
            'contact_info.fullName' => 'required|string',
            'contact_info.email' => 'required|email',
            'contact_info.phone' => 'required|string',
            'payment_type' => 'required|in:full,installment,partial',
            'cancellation_policy' => 'required|in:none,partial,full',
            'flight_preference' => 'nullable|string',
            'flight_type' => 'nullable|string',
            'airline' => 'nullable|string',
            'custom_flight_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $tour = Tour::findOrFail($request->tour_id);
            $tourDate = TourDate::findOrFail($request->tour_date_id);

            // Check availability
            $totalGuests = count($request->guests);
            if ($tourDate->available_spots < $totalGuests) {
                return response()->json([
                    'message' => 'Not enough available spots'
                ], 422);
            }

            // Calculate total price
            $totalPrice = 0;
            foreach ($request->guests as $guestData) {
                $price = match($guestData['type']) {
                    'adult', 'student' => $tourDate->adult_discounted_price ?? $tourDate->adult_price,
                    'child' => $tourDate->child_discounted_price ?? $tourDate->child_price ?? 0,
                    'baby' => $tourDate->baby_discounted_price ?? $tourDate->baby_price ?? 0,
                    default => 0,
                };
                $totalPrice += $price;
            }

            // Apply coupon if provided
            $discountAmount = 0;
            if ($request->has('coupon_code')) {
                $coupon = \App\Models\Coupon::where('code', $request->coupon_code)->first();
                if ($coupon && $coupon->isValid($totalPrice)) {
                    if ($coupon->discount_type === 'percentage') {
                        $discountAmount = ($totalPrice * $coupon->discount_value) / 100;
                    } else {
                        $discountAmount = $coupon->discount_value;
                    }
                    $totalPrice -= $discountAmount;
                }
            }

            // Generate reservation code (Unified & Dynamic)
            $code = Reservation::generateUnifiedCode($tour->brand_id);
            $reservationCode = $code;
            $easyCode = $code;

            // Uçuş bilgileri: request'ten flight_* alanları veya flight nesnesi
            $flightPref = $request->flight_preference
                ?? (is_array($request->flight) ? ($request->flight['preference'] ?? null) : null);
            $flightType = $request->flight_type
                ?? (is_array($request->flight) ? ($request->flight['type'] ?? null) : null);
            $airline = $request->airline
                ?? (is_array($request->flight) ? ($request->flight['airline'] ?? null) : null);
            $customFlightNotes = $request->custom_flight_notes
                ?? (is_array($request->flight) ? ($request->flight['custom_notes'] ?? null) : null);

            // Create reservation
            $reservation = Reservation::create([
                'user_id' => $request->user()?->id,
                'brand_id' => $tour->brand_id,
                'tour_id' => $tour->id,
                'tour_date_id' => $tourDate->id,
                'reservation_code' => $reservationCode,
                'easy_code' => $easyCode,
                'status' => 'pending',
                'total_price' => $totalPrice,
                'paid_amount' => 0,
                'remaining_amount' => $totalPrice,
                'discount_amount' => $discountAmount,
                'payment_type' => $request->payment_type,
                'cancellation_policy' => $request->cancellation_policy,
                'flight_preference' => $flightPref,
                'flight_type' => $flightType,
                'airline' => $airline,
                'custom_flight_notes' => $customFlightNotes,
                'contact_info' => $request->contact_info,
                'coupon_id' => $coupon->id ?? null,
                'channel' => 'website',
            ]);

            // Create rooms
            foreach ($request->rooms as $roomData) {
                $room = Room::create([
                    'reservation_id' => $reservation->id,
                    'room_type' => $roomData['roomType'] ?? 'double',
                    'adults' => $roomData['adults'] ?? 0,
                    'students' => $roomData['students'] ?? 0,
                    'children' => $roomData['children'] ?? 0,
                    'babies' => $roomData['babies'] ?? 0,
                    'hotel_type' => $roomData['hotelType'] ?? null,
                ]);

                // Assign guests to rooms
                $roomGuests = $roomData['guest_ids'] ?? [];
                foreach ($roomGuests as $index => $guestIndex) {
                    if (isset($request->guests[$guestIndex])) {
                        Guest::create([
                            'reservation_id' => $reservation->id,
                            'room_id' => $room->id,
                            'type' => $request->guests[$guestIndex]['type'],
                            'first_name' => $request->guests[$guestIndex]['first_name'],
                            'last_name' => $request->guests[$guestIndex]['last_name'],
                            'gender' => $request->guests[$guestIndex]['gender'] ?? null,
                            'date_of_birth' => $request->guests[$guestIndex]['date_of_birth'] ?? null,
                            'pickup_point_id' => $request->guests[$guestIndex]['pickup_point_id'] ?? null,
                            'seat_number' => $request->guests[$guestIndex]['seat_number'] ?? null,
                        ]);
                    }
                }
            }

            // Update tour date booked count
            $tourDate->increment('booked', $totalGuests);
            if ($tourDate->booked >= $tourDate->capacity) {
                $tourDate->update(['status' => 'full']);
            } elseif ($tourDate->occupancy_percentage >= 80) {
                $tourDate->update(['status' => 'limited']);
            }

            DB::commit();

            return response()->json([
                'message' => 'Reservation created successfully',
                'reservation' => [
                    'id' => $reservation->id,
                    'reservation_code' => $reservation->reservation_code,
                    'easy_code' => $reservation->easy_code,
                    'status' => $reservation->status,
                    'total_price' => $reservation->total_price,
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Reservation creation failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user reservations
     */
    public function index(Request $request): JsonResponse
    {
        $reservations = Reservation::with(['tour', 'tourDate', 'payments'])
            ->where('user_id', $request->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));

        return response()->json($reservations);
    }

    /**
     * Get reservation by ID
     */
    public function show(Request $request, int $id): JsonResponse
    {
        $reservation = Reservation::with([
            'tour', 'tourDate', 'guests', 'rooms', 'payments', 'documents', 'coupon'
        ])->findOrFail($id);

        // Check if user owns this reservation
        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        return response()->json([
            'reservation' => $reservation
        ]);
    }

    /**
     * Cancel reservation
     */
    public function cancel(Request $request, int $id): JsonResponse
    {
        $reservation = Reservation::findOrFail($id);

        if ($reservation->user_id !== $request->user()->id && !$request->user()->isAdmin()) {
            return response()->json([
                'message' => 'Unauthorized'
            ], 403);
        }

        if (!in_array($reservation->status, ['pending', 'approved', 'payment_pending'])) {
            return response()->json([
                'message' => 'Reservation cannot be cancelled'
            ], 422);
        }

        $reservation->update(['status' => 'cancelled']);

        // Refund tour date booked count
        $totalGuests = $reservation->guests()->count();
        $reservation->tourDate->decrement('booked', $totalGuests);

        return response()->json([
            'message' => 'Reservation cancelled successfully'
        ]);
    }

    /**
     * Get reservation status
     */
    public function status(Request $request, int $id): JsonResponse
    {
        $reservation = Reservation::findOrFail($id);

        return response()->json([
            'status' => $reservation->status,
            'reservation_code' => $reservation->reservation_code,
            'easy_code' => $reservation->easy_code,
            'total_price' => $reservation->total_price,
            'paid_amount' => $reservation->paid_amount,
            'remaining_amount' => $reservation->remaining_amount,
        ]);
    }
}
