<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Brand;
use App\Models\Tour;
use App\Models\TourDate;
use App\Models\Reservation;
use App\Models\Payment;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create roles
        $adminRole = Role::firstOrCreate(['name' => 'admin']);
        $superAdminRole = Role::firstOrCreate(['name' => 'super_admin']);
        $userRole = Role::firstOrCreate(['name' => 'user']);

        // Create admin user
        $admin = User::firstOrCreate(
            ['email' => 'admin@avusturyagezi.com'],
            [
                'name' => 'Admin User',
                'password' => Hash::make('password'),
                'user_type' => 'super_admin',
                'is_active' => true,
            ]
        );
        $admin->assignRole($superAdminRole);

        // Create test brand
        $brand = Brand::firstOrCreate(
            ['slug' => 'avusturya-gezi'],
            [
                'name' => 'Avusturya Gezi',
                'logo' => null,
                'domain' => 'avusturyagezi.com',
                'is_active' => true,
            ]
        );

        // Create test tour
        $tour = Tour::firstOrCreate(
            ['code' => 'BAL-001'],
            [
                'brand_id' => $brand->id,
                'title' => 'Bali & Singapur Turu',
                'type' => 'flight',
                'nights' => 7,
                'is_active' => true,
            ]
        );

        // Create test tour date
        $tourDate = TourDate::firstOrCreate(
            [
                'tour_id' => $tour->id,
                'start_date' => '2026-02-15',
            ],
            [
                'end_date' => '2026-02-22',
                'adult_price' => 1250.00,
                'child_price' => 900.00,
                'baby_price' => 0.00,
                'capacity' => 50,
                'booked' => 0,
                'status' => 'available',
            ]
        );

        // Create test customer
        $customer = User::firstOrCreate(
            ['email' => 'customer@example.com'],
            [
                'name' => 'Ahmet Yılmaz',
                'password' => Hash::make('password'),
                'user_type' => 'customer',
                'is_active' => true,
            ]
        );
        $customer->assignRole($userRole);

        // Create test reservation
        $reservation = Reservation::firstOrCreate(
            ['reservation_code' => 'RSV-TEST-001'],
            [
                'brand_id' => $brand->id,
                'user_id' => $customer->id,
                'tour_id' => $tour->id,
                'tour_date_id' => $tourDate->id,
                'easy_code' => 'AVG-001234',
                'status' => 'pending',
                'total_price' => 1250.00,
                'paid_amount' => 0.00,
                'remaining_amount' => 1250.00,
                'contact_info' => [
                    'fullName' => 'Ahmet Yılmaz',
                    'email' => 'customer@example.com',
                    'phone' => '+43 123 456 7890',
                ],
            ]
        );

        // Run Dubai Tour Seeder
        $this->call(DubaiTourSeeder::class);

        // Örnek yemekleri 15.02.2026 tarihli tura ekle (DENEME TUR veya ilk eşleşen tarih)
        $this->call(MealOptionsExampleSeeder::class);

        $this->command->info('Database seeded successfully!');
        $this->command->info('Admin: admin@avusturyagezi.com / password');
        $this->command->info('Customer: customer@example.com / password');
    }
}
