<?php

$requirements = [
    'php_version' => '8.1.0',
    'extensions' => [
        'bcmath',
        'ctype',
        'fileinfo',
        'json',
        'mbstring',
        'openssl',
        'pdo',
        'pdo_mysql',
        'tokenizer',
        'xml',
        'curl',
        'gd'
    ],
    'directories' => [
        '../storage',
        '../bootstrap/cache',
    ]
];

$results = [
    'php' => [],
    'extensions' => [],
    'directories' => [],
    'summary' => 'success'
];

// PHP Version Check
$currentPhpVersion = phpversion();
if (version_compare($currentPhpVersion, $requirements['php_version'], '>=')) {
    $results['php'] = [
        'status' => 'ok',
        'message' => "PHP Version: $currentPhpVersion (>= {$requirements['php_version']})"
    ];
} else {
    $results['php'] = [
        'status' => 'error',
        'message' => "PHP Version: $currentPhpVersion (Required: >= {$requirements['php_version']})"
    ];
    $results['summary'] = 'error';
}

// Extensions Check
foreach ($requirements['extensions'] as $ext) {
    if (extension_loaded($ext)) {
        $results['extensions'][$ext] = 'ok';
    } else {
        $results['extensions'][$ext] = 'missing';
        $results['summary'] = 'error';
    }
}

// Directory Permissions Check
// Note: We are in 'public/', so 'storage/' is '../storage/'
foreach ($requirements['directories'] as $dir) {
    $path = __DIR__ . '/' . $dir;
    if (!file_exists($path)) {
        // Try creating it? No, just report.
        $results['directories'][$dir] = 'missing';
        // $results['summary'] = 'error'; // Missing dir might just be empty repo, not server capability
    } elseif (is_writable($path)) {
        $results['directories'][$dir] = 'writable';
    } else {
        $results['directories'][$dir] = 'not_writable';
        $results['summary'] = 'error';
    }
}

// Output HTML
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Server Requirement Check</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif; background: #f8fafc; color: #334155; line-height: 1.5; padding: 2rem; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 2rem; border-radius: 12px; box-shadow: 0 4px 6px -1px rgb(0 0 0 / 0.1); }
        h1 { font-size: 1.5rem; font-weight: 700; margin-bottom: 2rem; color: #0f172a; text-align: center; }
        .section { margin-bottom: 2rem; }
        .section-title { font-size: 1.1rem; font-weight: 600; margin-bottom: 1rem; border-bottom: 2px solid #e2e8f0; padding-bottom: 0.5rem; }
        .item { display: flex; justify-content: space-between; align-items: center; padding: 0.75rem 0; border-bottom: 1px solid #f1f5f9; }
        .item:last-child { border-bottom: none; }
        .badge { padding: 0.25rem 0.75rem; border-radius: 9999px; font-size: 0.875rem; font-weight: 500; }
        .badge.ok, .badge.writable { background: #dcfce7; color: #166534; }
        .badge.error, .badge.missing, .badge.not_writable { background: #fee2e2; color: #991b1b; }
        .summary { text-align: center; padding: 1.5rem; margin-top: 2rem; border-radius: 8px; font-weight: bold; }
        .summary.success { background: #dcfce7; color: #166534; }
        .summary.error { background: #fee2e2; color: #991b1b; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Server Requirements Check</h1>

        <!-- PHP Version -->
        <div class="section">
            <h2 class="section-title">PHP Version</h2>
            <div class="item">
                <span>Current Version</span>
                <span class="badge <?php echo $results['php']['status']; ?>">
                    <?php echo $results['php']['message']; ?>
                </span>
            </div>
        </div>

        <!-- Extensions -->
        <div class="section">
            <h2 class="section-title">Required Extensions</h2>
            <?php foreach ($results['extensions'] as $ext => $status): ?>
            <div class="item">
                <span><?php echo $ext; ?></span>
                <span class="badge <?php echo $status == 'ok' ? 'ok' : 'error'; ?>">
                    <?php echo $status == 'ok' ? 'Installed' : 'Missing'; ?>
                </span>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Directory Permissions -->
        <div class="section">
            <h2 class="section-title">Directory Permissions</h2>
            <div class="item" style="display:block; font-size: 0.85rem; color: #64748b; margin-bottom: 10px;">
                Note: These directories must be writable by the web server user. Paths are relative to <code>public/</code>.
            </div>
            <?php foreach ($results['directories'] as $dir => $status): ?>
            <div class="item">
                <span><?php echo $dir; ?></span>
                <span class="badge <?php echo $status; ?>">
                    <?php echo $status == 'writable' ? 'Writable' : 'Not Writable / Missing'; ?>
                </span>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Summary -->
        <?php if ($results['summary'] === 'success'): ?>
        <div class="summary success">
            ✅ All requirements are met! Your server is ready for the backend.
        </div>
        <?php else: ?>
        <div class="summary error">
            ❌ Some requirements are missing. Please check the items above.
        </div>
        <?php endif; ?>

        <div style="margin-top: 20px; text-align: center; font-size: 0.9em; color: #666;">
            Generated by Antigravity Agent
        </div>
    </div>
</body>
</html>
